/*************************************************************************
 * This program is an adapted version of the FastLED library ColorPalette
 * example. This sketch programs TinyLily RGB LEDs and a TinyLily Accelerometer 
 * to display a Red, White, and Blue Color Palette based on movement.
 * 
 * Program initiated: 6-19-2019
 * Program last updated: 6-20-2019
 * 
 * Written by: Laveréna Wienclaw for TinyCircuits
 *************************************************************************/
 
// Some notes on the more abstract 'theory and practice' of
// FastLED compact palettes are at the bottom of this file.

#include <FastLED.h>
#include <Wire.h>         // For I2C communication with sensor
#include "BMA250.h"       // For interfacing with the accel. sensor

#define LED_PIN     3
#define NUM_LEDS    50
#define BRIGHTNESS  64 // 64/255 is 25% brightness
#define LED_TYPE    WS2811
#define COLOR_ORDER GRB
CRGB leds[NUM_LEDS];

#define UPDATES_PER_SECOND 250


CRGBPalette16 currentPalette;
TBlendType    currentBlending;

extern CRGBPalette16 myRedWhiteBluePalette;
extern const TProgmemPalette16 myRedWhiteBluePalette_p PROGMEM;

#if defined(ARDUINO_ARCH_SAMD)
 #define SerialMonitorInterface SerialUSB
#else
 #define SerialMonitorInterface Serial
#endif

const int powerPin = 4;
//
//// Accelerometer sensor variables for the sensor and its values
BMA250 accel_sensor;
int x, y, z;
double temp;

void setup() {
  SerialMonitorInterface.begin(115200);
  Wire.begin();
  delay(500);
//  while(!SerialMonitorInterface);

  // Setting variables to ouput digitally
  pinMode(powerPin, OUTPUT);
  digitalWrite(powerPin, HIGH);

  SerialMonitorInterface.print("Initializing BMA...");

  delay( 3000 ); // power-up safety delay

  accel_sensor.begin(BMA250_range_2g, BMA250_update_time_64ms); 
    
    FastLED.addLeds<LED_TYPE, LED_PIN, COLOR_ORDER>(leds, NUM_LEDS).setCorrection( TypicalLEDStrip );
    FastLED.setBrightness(  BRIGHTNESS );
    pinMode(LED_PIN, OUTPUT);
    
}

uint8_t brightness = 255;
int mostRecent_x;

void loop()
{

  accel_sensor.read();//This function gets new data from the acccelerometer

  // Get the acceleration values from the sensor and store them into global variables
  // (Makes reading the rest of the program easier)
  x = accel_sensor.X;
  y = accel_sensor.Y;
  z = accel_sensor.Z;
  temp = ((accel_sensor.rawTemp * 0.5) + 24.0);
    

  // If enough difference in x value is detected, change color
  // (There is no crazy science to why I used x or 30, it's just what looked best to me)
  if (abs(mostRecent_x - x) > 30)
  {
    currentPalette = myRedWhiteBluePalette_p; currentBlending = LINEARBLEND;
    
    static uint8_t startIndex = 0;
    startIndex = startIndex + 1; /* motion speed */
    
    for( int i = 0; i < NUM_LEDS; i++) {
        leds[i] = ColorFromPalette( currentPalette, startIndex, brightness, currentBlending);
        startIndex += 1; // Increase number to increase speed
    }
    
    FastLED.show();
    FastLED.delay(100 / UPDATES_PER_SECOND);
  }

  mostRecent_x = x; // update global variable

  // ***This delay is necessary to give the Accelerometer time to take a reading***
  delay(250);
}


// This example shows how to set up a static color palette
// which is stored in PROGMEM (flash), which is almost always more
// plentiful than RAM.  A static PROGMEM palette like this
// takes up 64 bytes of flash.
const TProgmemPalette16 myRedWhiteBluePalette_p PROGMEM =
{
    CRGB::Red,
    CRGB::Gray, // 'White' is too bright compared to red and blue
    CRGB::Blue,
    CRGB::Black,
    
    CRGB::Red,
    CRGB::Gray,
    CRGB::Blue,
    CRGB::Black,
    
    CRGB::Red,
    CRGB::Red,
    CRGB::Gray,
    CRGB::Gray,
    CRGB::Blue,
    CRGB::Blue,
    CRGB::Black,
    CRGB::Black
};



// Additional notes on FastLED compact palettes:
//
// Normally, in computer graphics, the palette (or "color lookup table")
// has 256 entries, each containing a specific 24-bit RGB color.  You can then
// index into the color palette using a simple 8-bit (one byte) value.
// A 256-entry color palette takes up 768 bytes of RAM, which on Arduino
// is quite possibly "too many" bytes.
//
// FastLED does offer traditional 256-element palettes, for setups that
// can afford the 768-byte cost in RAM.
//
// However, FastLED also offers a compact alternative.  FastLED offers
// palettes that store 16 distinct entries, but can be accessed AS IF
// they actually have 256 entries; this is accomplished by interpolating
// between the 16 explicit entries to create fifteen intermediate palette
// entries between each pair.
//
// So for example, if you set the first two explicit entries of a compact 
// palette to Green (0,255,0) and Blue (0,0,255), and then retrieved 
// the first sixteen entries from the virtual palette (of 256), you'd get
// Green, followed by a smooth gradient from green-to-blue, and then Blue.
